// DarkVIDEO (C) Deep Cove Software, Ltd. - www.deepcovesoftware.com
// Dark GDK (C) The Game Creators - www.thegamecreators.com

// this sample is based on the Dark Video DarkBASIC Professional sample,
// which in turn is based on The Game Creators 3DVideo sample.

// whenever using Dark GDK you must ensure you include the header file
#include "DarkGDK.h"

// include the header file for DarkVIDEO
#include "DarkVIDEO.h"

// the main entry point for the application is this function
void DarkGDK ( void )
{
	// initialise DarkVIDEO.
	DarkVideoInitialize();

	// when starting a Dark GDK program it is useful to set global
	// application properties, we begin by turning the sync rate on,
	// this means we control when the screen is updated, we also set
	// the maximum rate to 30 which means the maximum frame rate will
	// be set at 30 frames per second.
	dbSyncOn   ( );
	dbSyncRate ( 30 );

	// set the window title.
	dbSetWindowTitle( "DarkVIDEO DarkGDK Sample" );

	// set default screen.
	dbColorBackdrop( dbRGB( 0, 32, 32 ) );

	// open our DarkVIDEO movie (*.dvf) and get the dimensions (we should
	// check the return value of DarkVideoOpen here but for simplicity...).
	int movieId = DarkVideoOpen( "sample.dvf" );
	int movieWidth = DarkVideoGetWidth( movieId );
	int movieHeight = DarkVideoGetHeight( movieId );

	// set the movie volume (between 0-255), no effect of course if there's
	// no audio in the movie.
	DarkVideoSetVolume( movieId, 255 );

	// arbitrary ids we'll use.
	int memBlockId = 32;
	int imageId = 32;

	// create a blank 32-bit image of the same dimensions as the DarkVIDEO
	// movie. DarkVIDEO will decode each frame to this image.
	// the size is the width by the height by 4 bytes per pixel as it's
	// 32-bit colour. we also add 12 bytes for 3 dwords at the start of
	// the memblock that contain the width, height and bit depth as we're
	// going to create an image from this memblock.
	dbMakeMemblock( memBlockId, movieWidth * movieHeight * 4 + 12 );
	dbWriteMemblockDword( memBlockId, 0, movieWidth );
	dbWriteMemblockDword( memBlockId, 4, movieHeight );
	dbWriteMemblockDword( memBlockId, 8, 32 );
	dbMakeImageFromMemblock( imageId, memBlockId );
	dbDeleteMemblock( memBlockId );

	// create 3D projection.
	for ( int p = 1; p <= 3; p++ )
	{
		dbMakeObjectBox( p, 320, 240, 10 );
		dbScaleObject( p, p * 250, p * 250, p * 250 );
		if ( p > 1 )
			dbGhostObjectOn( p, 2 );
		dbTextureObject( p, imageId );
	}
	dbSetObjectLight( 1, 0 );

	// setup camera.
	dbPositionCamera( 0, 0, -700 );
	dbRotateCamera( 0, 0, 0 );

	bool begin = true;
	float a = 0.0f;

	// now we come to our main loop, we call LoopGDK so some internal
	// work can be carried out by the GDK.
	while ( LoopGDK ( ) )
	{
		// begin movie playback.
		if ( begin )
		{
			begin = false;
			bool loop = true;

			// note the last parameter? This *HAS* to be set to dbGetImagePointer
			// or DarkVIDEO is unable to decode the movie to a DarkGDK image.
			DarkVideoPlay( movieId, imageId, loop, dbGetImagePointer );
		}

		// move projection around.
		for ( int p = 1; p <= 3; p++ )
		{
			a = dbWrapValue( a + 1.0f );
			dbPositionObject( p, dbCOS( a ) * 20, dbSIN( a ) * 10, p * 75 );
			dbYRotateObject( p, dbCOS( a ) * 20 );
			dbZRotateObject( p, ( p - 1 ) * 10 );
		}

		// show the current frame.
		char buffer[ 256 ];
		sprintf( buffer, "Current Frame: %d", DarkVideoGetCurrentFrame( movieId ) );
		dbText( 0, 0, buffer );

		// show the number of frames.
		sprintf( buffer, "Number Of Frames: %d", DarkVideoGetNumberOfFrames( movieId ) );
		dbText( 0, 15, buffer );

		// show the frame rate.
		sprintf( buffer, "Movie Framerate: %.2f", DarkVideoGetFrameRate( movieId ) );
		dbText( 0, 30, buffer );

		// show the movie width.
		sprintf( buffer, "Movie Width: %d", DarkVideoGetWidth( movieId ) );
		dbText( 0, 45, buffer );

		// show the movie height.
		sprintf( buffer, "Movie Height: %d", DarkVideoGetHeight( movieId ) );
		dbText( 0, 60, buffer );

		// is the movie finished?
		sprintf( buffer, "Movie Finished: %s", ( DarkVideoIsFinished( movieId ) ) ? "YES" : "NO" );
		dbText( 0, 75, buffer );

		// is the movie looping?
		sprintf( buffer, "Movie Looping: %s", ( DarkVideoIsLooping( movieId ) ) ? "YES" : "NO" );
		dbText( 0, 90, buffer );
		
		// here we make a call to update the contents of the screen.
		dbSync ( );
	}

	// stop and shutdown DarkVIDEO movie.
	DarkVideoStop( movieId );
	DarkVideoClose( movieId );

	// shutdown DarkVIDEO.
	DarkVideoShutdown();

	// and now everything is ready to return back to Windows.
	return;
}